package ru.bitel.bgbilling.modules.inet.dyn.device.misc;

import java.net.InetAddress;
import java.net.SocketException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;

import ru.bitel.bgbilling.kernel.network.radius.RadiusAttributeSet;
import ru.bitel.bgbilling.kernel.network.radius.RadiusClient;
import ru.bitel.bgbilling.kernel.network.radius.RadiusPacket;
import ru.bitel.bgbilling.modules.inet.access.sa.ServiceActivator;
import ru.bitel.bgbilling.modules.inet.access.sa.ServiceActivatorAdapter;
import ru.bitel.bgbilling.modules.inet.access.sa.ServiceActivatorEvent;
import ru.bitel.bgbilling.modules.inet.api.common.bean.InetConnection;
import ru.bitel.bgbilling.modules.inet.api.common.bean.InetDevice;
import ru.bitel.bgbilling.modules.inet.api.common.bean.InetDeviceType;
import ru.bitel.bgbilling.server.util.Setup;
import ru.bitel.common.ParameterMap;
import ru.bitel.common.Utils;
import ru.bitel.common.inet.IpAddress;
import ru.bitel.common.util.MacrosFormat;

/**
 * Генератор RADIUS-accounting-start/stop для IPoE/DHCP-схем.
 * 
 * sa.radius.fanout.accountingStart.attributes.macros=User-Name=$contractTitle();Acct-Session-Id=$connectionId()-$acctSessionId();Acct-Status-Type=1;Framed-IP-Address=$ip();Calling-Station-Id=$callingStationId()
 * sa.radius.fanout.accountingStop.attributes.macros=User-Name=$contractTitle();Acct-Session-Id=$connectionId()-$acctSessionId();Acct-Status-Type=2;Framed-IP-Address=$ip();Calling-Station-Id=$callingStationId();Acct-Terminate-Cause=1
 * 
 * Исключить сервисы (например, те, что в dhcp.disable.servId)
 * sa.radius.fanout.skipServIds=
 * 
 * @author amir
 */
public class RadiusFanoutServiceActivator
	extends ServiceActivatorAdapter
	implements ServiceActivator
{
	private static final Logger logger = Logger.getLogger( RadiusFanoutServiceActivator.class );

	private class RadiusFanoutMacros
		extends MacrosFormat
	{
		private final String patternString;

		public RadiusFanoutMacros( String pattern )
		{
			this.patternString = pattern;
		}

		@Override
		public StringBuffer format( StringBuffer appendable, Object... globalArgs )
		{
			return super.format( appendable, patternString, globalArgs );
		}

		@Override
		protected Object invoke( final String macros, final Object[] args, final Object[] globalArgs )
		{
			final ServiceActivatorEvent event = (ServiceActivatorEvent)globalArgs[0];
			final InetConnection connection = event.getConnection();

			if( "contractTitle".equals( macros ) )
			{
				String contractTitle = event.getInetServRuntime().contractRuntime.getContractTitle();
				if( Utils.notBlankString( contractTitle ) )
				{
					return contractTitle;
				}

				return String.valueOf( event.getInetServRuntime().contractRuntime.contractId );
			}
			else if( "ip".equals( macros ) )
			{
				byte[] address = connection.getInetAddressBytes();
				if( address == null )
				{
					address = event.getInetServRuntime().getInetServ().getAddressFrom();
				}

				return IpAddress.toString( address );
			}
			else if( "contractId".equals( macros ) )
			{
				return String.valueOf( event.getInetServRuntime().contractRuntime.contractId );
			}
			else if( "inetServId".equals( macros ) )
			{
				return String.valueOf( event.getInetServRuntime().inetServId );
			}
			else if( "connectionId".equals( macros ) )
			{
				return connection.getId();
			}
			else if( "callingStationId".equals( macros ) )
			{
				return connection.getCallingStationId();
			}
			else if( "acctSessionId".equals( macros ) )
			{
				return connection.getAcctSessionId();
			}
			else if( "deviceId".equals( macros ) )
			{
				return connection.getDeviceId();
			}
			else if( "agentDeviceId".equals( macros ) )
			{
				return connection.getAgentDeviceId();
			}
			else if( "circuitId".equals( macros ) )
			{
				return connection.getCircuitId();
			}

			return connection.getId();
		}
	}

	private int deviceId;

	private Set<Integer> skipServIds;

	/**
	 * Radius-клиент
	 */
	protected RadiusClient radiusClient;

	protected RadiusFanoutMacros accountingStartMacros;
	protected RadiusFanoutMacros accountingStopMacros;

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object init( Setup setup, int moduleId, InetDevice device, InetDeviceType deviceType, ParameterMap deviceConfig )
		throws Exception
	{
		this.deviceId = device.getId();

		final List<String[]> hosts = device.getHostsAsString();
		final String[] host = (hosts != null && hosts.size() > 0) ? hosts.get( 0 ) : null;

		String nasHost = deviceConfig.get( "sa.radius.fanout.host", host != null ? host[0] : device.getHost() );
		InetAddress nasHostAddr = InetAddress.getByName( nasHost );

		int nasPort = deviceConfig.getInt( "sa.radius.fanout.port", Utils.parseInt( host != null ? host[1] : "1700" ) );

		if( nasPort <= 0 )
		{
			nasPort = 1700;
		}

		String sourceHost = deviceConfig.get( "sa.radius.sourceHost", deviceConfig.get( "radius.sourceHost", null ) );
		int sourcePort = deviceConfig.getInt( "sa.radius.sourcePort", deviceConfig.getInt( "radius.sourcePort", -1 ) );

		byte[] nasSecret = deviceConfig.get( "sa.radius.fanout.secret",
											 deviceConfig.get( "sa.radius.secret",
															   deviceConfig.get( "radius.secret", device.getSecret() ) ) )
									   .getBytes();

		this.accountingStartMacros =
			new RadiusFanoutMacros( deviceConfig.get( "sa.radius.fanout.accountingStart.attributes.macros",
													  "User-Name=$contractTitle();Acct-Session-Id=$connectionId()-$acctSessionId();Acct-Status-Type=1;Framed-IP-Address=$ip();Calling-Station-Id=$callingStationId()" ) );
		this.accountingStopMacros =
			new RadiusFanoutMacros( deviceConfig.get( "sa.radius.fanout.accountingStop.attributes.macros",
													  "User-Name=$contractTitle();Acct-Session-Id=$connectionId()-$acctSessionId();Acct-Status-Type=2;Framed-IP-Address=$ip();Calling-Station-Id=$callingStationId();Acct-Terminate-Cause=1" ) );

		this.skipServIds = Utils.toIntegerSet( deviceConfig.get( "sa.radius.fanout.skipServIds", null ) );

		this.radiusClient = new RadiusClient( sourceHost, sourcePort, nasHostAddr, nasPort, nasSecret );

		return null;
	}

	@Override
	public Object destroy()
		throws Exception
	{
		radiusClient.destroy();

		return null;
	}

	private Future<Boolean> sendRequest( final ServiceActivatorEvent event, final RadiusFanoutMacros macros )
		throws InvalidKeyException, SocketException, NoSuchAlgorithmException
	{
		if( skipServIds.contains( event.getInetServId() ) )
		{
			logger.debug( "Skip fanout accounting event" );
			return null;
		}

		final String macrosRadiusAttributes = macros.format( new StringBuffer( 200 ), event ).toString();

		final RadiusPacket request = radiusClient.createAccountingRequest();

		request.addAttributes( RadiusAttributeSet.newRadiusAttributeSet( macrosRadiusAttributes ) );

		logger.info( "Send fanout request: \n" + request );

		return radiusClient.sendAsync( request );
	}

	@Override
	public Object onAccountingStart( ServiceActivatorEvent event )
		throws Exception
	{
		sendRequest( event, accountingStartMacros );

		return null;
	}

	@Override
	public Object onAccountingStop( ServiceActivatorEvent event )
		throws Exception
	{
		sendRequest( event, accountingStopMacros );

		return null;
	}
}
