package ru.bitel.bgbilling.modules.inet.dyn.device.terminal;

import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

import ru.bitel.bgbilling.modules.inet.access.sa.ServiceActivator;
import ru.bitel.bgbilling.modules.inet.access.sa.ServiceActivatorEvent;
import ru.bitel.bgbilling.modules.inet.api.common.bean.InetConnection;
import ru.bitel.bgbilling.modules.inet.api.common.bean.InetDevice;
import ru.bitel.bgbilling.modules.inet.api.common.bean.InetDeviceType;
import ru.bitel.bgbilling.modules.inet.api.common.bean.InetServ;
import ru.bitel.bgbilling.server.util.Setup;
import ru.bitel.common.ParameterMap;
import ru.bitel.common.Utils;
import bitel.billing.server.admin.errorlog.AlarmSender;
import bitel.billing.server.admin.errorlog.bean.AlarmErrorMessage;
import bitel.billing.server.util.ssh.SSHSession;

/**
 *  {@inheritDoc}
 *  @see AbstractTerminalServiceActivator
 */
public class SSHServiceActivator
    extends AbstractTerminalServiceActivator
    implements ServiceActivator
{
	private static final Logger logger = Logger.getLogger( SSHServiceActivator.class );

	/**
	 *  Regexp как признак выхода
	 */
	protected String regexp;

	protected String endSequence;

	protected SSHSession session;

	protected boolean lazyConnect;
	
	@Override
	public Object init( Setup setup, int moduleId, InetDevice device, InetDeviceType deviceType, ParameterMap config )
	    throws Exception
	{
		super.init( setup, moduleId, device, deviceType, config );
		
		if( this.port <= 0 )
		{
			this.port = 22;
		}

		this.endSequence = config.get( "sa.endSequence", null );
		this.regexp = config.get( "sa.exitRegexp", null );

		this.lazyConnect = config.getInt( "sa.lazyConnect", 0 ) > 0;

		return null;
	}

	@Override
	public Object destroy()
	    throws Exception
	{
		return super.destroy();
	}

	@Override
	public Object connect()
		throws Exception
	{
		if( lazyConnect )
		{
			return null;
		}

		return connectImpl();
	}

	protected Object connectImpl()
		throws Exception
	{
		try
		{
			if( connectionSemaphore != null )
			{
				connectionSemaphoreAcquired = connectionSemaphore.tryAcquire( 3, TimeUnit.MINUTES );
			}
			
			SSHSession session = new SSHSession( host, port, username, password );
			session.setTimeout( timeout );
			session.setReadWait( readWait );

			if( Utils.notBlankString( endSequence ) )
			{
				session.setEndString( endSequence );
			}

			if( Utils.notBlankString( regexp ) )
			{
				session.setRegexp( regexp );
			}

			session.connect();
			logger.info( "Connected to " + device );

			this.session = session;
		}
		catch( Exception ex )
		{
			logger.error( "Can't connect to " + device + " to " + host + ":" + port + ". Check host/port, username/password and sa.endSequence (command prompt)" );
			throw ex;
		}

		return super.connect();
	}
	
	protected SSHSession getSession()
		throws Exception
	{
		if( session != null )
		{
			return session;
		}

		connectImpl();

		return session;
	}

	@Override
	public Object disconnect()
		throws Exception
	{
		try
		{
			if( session != null )
			{
				try
				{
					super.disconnect();

					logger.info( "[" + device + "] executeAsync: " + this.exitCommand );
					session.doCommandAsync( this.exitCommand );
				}
				finally
				{
					session.disconnect();

					session = null;

					logger.debug( "Disconnected" );
				}
			}
			else
			{
				logger.debug( "Not connected - skip disconnection" );
			}
		}
		finally
		{
			if( connectionSemaphore != null && connectionSemaphoreAcquired )
			{
				connectionSemaphore.release();
			}
		}

		return null;
	}
	
	private static Pattern returnCodePattern = Pattern.compile( "RETURN_CODE=(\\d+)" );

	@Override
	protected void executeCommand( String command )
		throws Exception
	{
		final SSHSession session = getSession();
		
		logger.info( "[" + device + "] execute: " + command );

		String result;
		try
		{
			result = session.doCommand( command );
		}
		catch( TimeoutException ex )
		{
			logger.error( "Timeout waiting command prompt (endSequence) when executing command." );
			throw ex;
		}

		logger.info( result );

		if( Utils.notBlankString( result ) )
		{
			Matcher m = returnCodePattern.matcher( result );
			while( m.find() )
			{
				String returnCode = m.group( 1 );

				if( Utils.parseInt( returnCode, 0 ) == 0 )
				{
					continue;
				}

				logger.error( "Command \"" + command + "\" executed with RETURN_CODE=" + returnCode );

				String key = "inet.sa.ssh.returnCode";
				long time = System.currentTimeMillis();

				if( AlarmSender.needAlarmSend( key, time, 600000 ) )
				{
					String subject = "Inet: Ошибка работы обработчика активации сервисов [" + deviceId + "] " + device;
					String message = "Ошибка работы обработчика активации сервисов [" + deviceId + "] "
									 + device
									 + "\n" + device.getHost()
									 + "\n(" + device.getComment() + ")\n\n"
									 + "Исполнение команды \"" + command + "\" завершилось с ошибкой " + returnCode + "!";

					AlarmErrorMessage alarm = new AlarmErrorMessage( key, subject, message );

					AlarmSender.sendAlarm( alarm, time );
				}
			}
		}
	}

	@Override
	protected Object getValue( ServiceActivatorEvent e, InetServ serv, InetConnection connection, Set<Integer> options, String macros, Object[] args, Object[] globalArgs )
	    throws Exception
	{
		if( "setEndSequence".equals( macros ) )
		{
			if( args.length > 0 )
			{
				String endSequence = (String)args[0];
				if( Utils.notEmptyString( endSequence ) )
				{
					getSession().setEndString( (String)args[0] );
					return "";
				}
			}

			getSession().setEndString( this.endSequence );
			return "";
		}
		else
		{
			return super.getValue( e, serv, connection, options, macros, args, globalArgs );
		}
	}
}
