package ru.bitel.bgbilling.modules.inet.dyn.device.terminal;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.Socket;
import java.net.URI;
import java.util.Scanner;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

import bitel.billing.server.util.ssh.SSHSession;
import bitel.billing.server.util.telnet.TelnetSession;
import ru.bitel.common.Utils;

public abstract class TerminalSession
	implements AutoCloseable
{
	private static final Logger logger = Logger.getLogger( TerminalSession.class );

	protected long timeout = 60000;
	protected long readWait = 50;

	protected String endSequence;
	protected String exitCommand = "exit";

	public void setTimeout( long timeout )
	{
		this.timeout = timeout;
	}

	public void setReadWait( long readWait )
	{
		this.readWait = readWait;
	}

	public void setEndSequence( String endSequence )
	{
		this.endSequence = endSequence;
	}

	public void setExitCommand( String exitCommand )
	{
		this.exitCommand = exitCommand;
	}

	public abstract void connect( String username, String password )
		throws Exception;

	public abstract boolean isConnected();

	public String sendCommand( String command )
		throws Exception
	{
		logger.info( "sendCommand: " + command );
		String result = sendCommandImpl( command );
		logger.info( "result: " + result );
		return result;
	}

	protected abstract String sendCommandImpl( String command )
		throws Exception;

	static class TelnetTerminalSession
		extends TerminalSession
	{
		private static final Logger logger = Logger.getLogger( TelnetTerminalSession.class );

		private final String host;
		private final int port;

		private TelnetSession session;

		public TelnetTerminalSession( String host, int port )
		{
			this.host = host;
			this.port = port;
		}

		@Override
		public void setEndSequence( String endSequence )
		{
			super.setEndSequence( endSequence );

			if( session != null )
			{
				session.setEndString( endSequence );
			}
		}

		@Override
		public void connect( String username, String password )
			throws Exception
		{
			if( this.session != null )
			{
				logger.error( "Already connected!" );
				return;
			}

			TelnetSession session = new TelnetSession( host, port );

			session.setTimeout( (int)timeout );
			session.setReadWait( readWait );

			session.setEndString( ":" );

			session.connect();
			logger.info( "Connected" );

			this.session = session;

			logger.info( session.doCommand( username ) );
			logger.info( "Login entered" );

			if( endSequence == null )
			{
				endSequence = "#";
			}

			session.setEndString( endSequence );

			logger.info( session.doCommand( password ) );
			logger.info( "Password entered" );
		}

		@Override
		protected String sendCommandImpl( String command )
			throws Exception
		{
			return session.doCommand( command );
		}

		@Override
		public void close()
			throws Exception
		{
			if( session != null )
			{
				try
				{
					if( Utils.notBlankString( this.exitCommand ) )
					{
						logger.info( "sendCommandA: " + this.exitCommand );
						session.doCommandAsync( this.exitCommand );
					}
				}
				finally
				{
					session.disconnect();

					session = null;

					logger.debug( "Disconnected" );
				}
			}
		}

		@Override
		public boolean isConnected()
		{
			return session != null;
		}
	}

	static class SshTerminalSession
		extends TerminalSession
	{
		private static final Logger logger = Logger.getLogger( SshTerminalSession.class );

		private final String host;
		private final int port;

		private SSHSession session;

		public SshTerminalSession( String host, int port )
		{
			this.host = host;
			this.port = port;
		}

		@Override
		public void setEndSequence( String endSequence )
		{
			super.setEndSequence( endSequence );

			if( session != null )
			{
				session.setEndString( endSequence );
			}
		}

		@Override
		public void connect( String username, String password )
			throws Exception
		{
			SSHSession session = new SSHSession( host, port, username, password );
			session.setTimeout( (int)timeout );
			session.setReadWait( readWait );

			if( Utils.notBlankString( endSequence ) )
			{
				session.setEndString( endSequence );
			}

			session.connect();
			logger.info( "Connected" );

			this.session = session;
		}

		@Override
		protected String sendCommandImpl( String command )
			throws Exception
		{
			return session.doCommand( command );
		}

		@Override
		public void close()
			throws Exception
		{
			if( session != null )
			{
				try
				{
					logger.info( "sendCommandA: " + this.exitCommand );
					session.doCommandAsync( this.exitCommand );
				}
				finally
				{
					session.disconnect();

					session = null;

					logger.debug( "Disconnected" );
				}
			}
		}

		@Override
		public boolean isConnected()
		{
			return session != null;
		}
	}

	static class TcpTerminalSession
		extends TerminalSession
	{
		private static final Logger logger = Logger.getLogger( TcpTerminalSession.class );

		private final String host;
		private final int port;

		private Socket socket;
		private PrintWriter out;
		private BufferedReader in;

		private Scanner scanner;

		private Pattern pattern;

		public TcpTerminalSession( String host, int port )
		{
			this.host = host;
			this.port = port;
		}

		@Override
		public void connect( String username, String password )
			throws Exception
		{
			socket = new Socket( host, port );

			socket.setSoTimeout( (int)timeout );

			out = new PrintWriter( socket.getOutputStream(), true );

			InputStreamReader isr = new InputStreamReader( socket.getInputStream() );
			in = new BufferedReader( isr );

			scanner = new Scanner( in );

			logger.info( "Connected" );
		}

		@Override
		public void setEndSequence( String endSequence )
		{
			super.setEndSequence( endSequence );

			pattern = Pattern.compile( Pattern.quote( endSequence ), Pattern.DOTALL );
		}

		@Override
		protected String sendCommandImpl( String command )
			throws Exception
		{
			out.println( command );
			out.flush();

			if( Utils.isEmptyString( endSequence ) )
			{
				return "";
			}

			scanner.useDelimiter( pattern );

			return scanner.next();
		}

		@Override
		public void close()
			throws Exception
		{
			if( out != null )
			{
				out.close();
				socket.close();

				scanner.close();

				logger.info( "Disconnected" );

				out = null;
				socket = null;
			}
		}

		@Override
		public boolean isConnected()
		{
			return out != null;
		}
	}

	public static TerminalSession newSshSession( String host, int port )
	{
		return new SshTerminalSession( host, port );
	}

	public static TerminalSession newTelnetSession( String host, int port )
	{
		return new TelnetTerminalSession( host, port );
	}

	public static TerminalSession newTcpSession( String host, int port )
	{
		return new TcpTerminalSession( host, port );
	}

	public static TerminalSession newTerminalSession( String type, String host, int port )
	{
		switch( type )
		{
			case "ssh":
				return new SshTerminalSession( host, port );

			case "tcp":
				return new TcpTerminalSession( host, port );

			case "telnet":
			default:
				return new TelnetTerminalSession( host, port );
		}
	}

	public static TerminalSession newTerminalSession( URI uri )
	{
		return newTerminalSession( uri.getScheme(), uri.getHost(), uri.getPort() );
	}
}
