package ru.bitel.bgbilling.modules.tv.dyn.netup;

import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import org.json.JSONArray;
import org.json.JSONObject;

import ru.bitel.bgbilling.apps.tv.access.om.AbstractOrderEvent;
import ru.bitel.bgbilling.apps.tv.access.om.AccountOrderEvent;
import ru.bitel.bgbilling.apps.tv.access.om.ProductOrderEvent;
import ru.bitel.bgbilling.kernel.base.server.logger.BGLogger;
import ru.bitel.bgbilling.kernel.container.managed.ServerContext;
import ru.bitel.bgbilling.modules.tv.common.bean.TvAccount;
import ru.bitel.bgbilling.modules.tv.common.bean.TvDevice;
import ru.bitel.bgbilling.modules.tv.common.bean.TvDeviceType;
import ru.bitel.bgbilling.modules.tv.common.om.OrderManager;
import ru.bitel.bgbilling.modules.tv.dyn.JsonClient;
import ru.bitel.bgbilling.modules.tv.dyn.JsonClient.JsonClientException;
import ru.bitel.common.ParameterMap;
import ru.bitel.common.Utils;
import ru.bitel.oss.systems.inventory.product.common.bean.ProductSpec;

/**
 * http://stand.netup.tv/downloads/mw-api.html
 * абонент авторизируется по логин/пасс
 * @author dimon
 */
public class NetUpOrderManager
    extends BGLogger
    implements OrderManager
{
    private JsonClient jsonClient;

    private NetUpConf conf;
 
    private Map<String, String> requestOptions = new HashMap<>();

    @Override
    public Object init( ServerContext ctx, int moduleId, TvDevice tvDevice, TvDeviceType tvDeviceType, ParameterMap config )
        throws Exception
    {
        this.conf = ctx.getSetup().getConfig( moduleId, NetUpConf.class );
        this.conf.fill(config);
        this.requestOptions.put( "Content-Type", "application/json" );
        this.requestOptions.put( "Accept", "application/json" );
        return null;
    }

	@Override
	public Object destroy()
		throws Exception
	{
		return null;
	}

	@Override
	public Object connect( ServerContext ctx )
		throws Exception
	{
		jsonClient = new JsonClient( URI.create( conf.providerURL ).toURL(), null, null );
		return null;
	}

    @Override
    public Object disconnect( ServerContext ctx )
        throws Exception
    {
        if ( jsonClient != null )
        {
            jsonClient.disconnect();
            jsonClient = null;
        }

        return null;
    }

    @Override
    public Object accountCreate( AccountOrderEvent accountOrderEvent, ServerContext serverContext )
        throws Exception
    {
        // вызов: создание аккаунта
        final TvAccount tvAccount = accountOrderEvent.getNewTvAccount();
        
        String login = tvAccount.getLogin();
        String password = tvAccount.getPassword();
        
        getLogger().info("NetUpOrderManager.accountCreate login=" + login);
        
        JSONObject account = new JSONObject();
        account.put( "login", login );
        account.put( "password", password );
        
        try
        {
            Object response = jsonClient.invokeAndGetObject( JsonClient.Method.post, requestOptions, conf.providerURL + "/user", null, account );
            // >> {"password":"653129","login":"10000003"}
            // << 5
            // возвращается просто число - access card
            if ( response != null && response instanceof Integer )
            {
                String access_card = response.toString();
                accountOrderEvent.getEntry().setDeviceAccountId(access_card);
                
                //tvAccount.setIdentifier( access_card );
                //tvAccountDao.update( tvAccount );
                //serverContext.publishAfterCommit( new TvAccountModifiedEvent( moduleId, accountOrderEvent.getContractId(), 0, child, newChild ) );
            }
            else
            {
                getLogger().error("accountCreate: unknown response: " + response);
                return false;
            }
        }
        catch( JsonClientException ex )
        {
            // Response code = 500
            // {"error":"Failed to insert record in customers"}
            if ( ex.getResponseCode() == 500 && Utils.maskNull(ex.getData()).contains("record in customers") )
            {
                getLogger().error( "accountCreate: duplicate login? response 500 : " + ex.getData() );
                return false;
            }
            throw ex;
        }
        
        return null;
    }

    private static String ACC_STATE_BLOCKED = "blocked";
    private static String ACC_STATE_ACTIVE = "active";
    
    @Override
    public Object accountModify(AccountOrderEvent accountOrderEvent, ServerContext serverContext) throws Exception
    {
        // вызов: редактирование аккаунта, в т.ч. внутри смена статуса аккаунта
        // вызов: сдвиг дат при редактировании на актуальные (при предыдущем редактировании могло быть вызвано удаление! см. accountRemove)
        final TvAccount tvAccount = accountOrderEvent.getNewTvAccount();
        final String access_card = tvAccount.getDeviceAccountId();
        
        getLogger().info("NetUpOrderManager.accountModify login=" + tvAccount.getLogin());
        
        if (Utils.isBlankString(access_card))
        {
            getLogger().error( "accountModify: access_card for account {id="+tvAccount.getId()+", login=" + tvAccount.getLogin() + "} is empty" );
            return false;
        }
        
        String state = switch (tvAccount.getStatus()) {
            case TvAccount.STATUS_CLOSED:
            case TvAccount.STATUS_LOCKED:
                yield ACC_STATE_BLOCKED;
            default:
                yield ACC_STATE_ACTIVE;
        };
        
        JSONObject account = new JSONObject();
        account.put( "state", state );
        
        /*JSONObject response =*/ jsonClient.invoke( JsonClient.Method.post, requestOptions, conf.providerURL + "/personal-account-block/" + access_card, null, account );
        // по протоколу ничего не возвращается, на деле - эхо
        // >> {"state":"blocked"}
        // << {"state":"blocked"}
        // Response code = 400
        // << {"error":"failed to parse URL parameter|name:access_card|raw_value:6_666"}
        
        // после этого надо ещё дропнуть подписку если заблочили, ибо mw нормально не работает тут, аккаунт
        // разблокируется сам после логина клиента туда(!), нетап отвечает следующее:
        // "Блокировка аккаунта - не означает блокировки услуг. Вы даете доступ к услугам и их и надо блокировать.
        // curl ***/access-media/34  -X POST -H "Content-Type: application/json" -d '{}'"
        // при разблокировке тоже синхронизируем (ну, при каждом редактировании аккаунта получается)
        if (!productsModifySyncFull(accountOrderEvent, serverContext, access_card, state.equals(ACC_STATE_ACTIVE) ))
        {
            getLogger().error( "accountModify: not succes productsModifySyncFull" );
            return false;
        }
        
        return null;
    }

    @Override
    public Object accountRemove(AccountOrderEvent accountOrderEvent, ServerContext serverContext) throws Exception
    {
        // вызов: сдвиг дат при редактировании на неактуальные. фактическое удаление (даёт только при неактульном периоде)
        final TvAccount tvAccount = accountOrderEvent.getOldTvAccount();
        final String access_card = tvAccount.getDeviceAccountId();
        
        getLogger().info("NetUpOrderManager.accountRemove login=" + tvAccount.getLogin());
       
        if (Utils.isBlankString(access_card))
        {
            getLogger().error( "accountRemove: access_card for account {id="+tvAccount.getId()+", login=" + tvAccount.getLogin() + "} is empty" );
            return false;
        }
            
        try
        {
            Object response = jsonClient.invokeAndGetObject( JsonClient.Method.delete, requestOptions, conf.providerURL + "/user/" + access_card, null, null );
            // по протоколу возвращается значение: true
            // но не получилось до сюда дойти, всегда 500
            getLogger().info("response=" + response); 
            
            if ( response == null || !(response instanceof Boolean) )
            {            
                getLogger().error("accountRemove: unknown response: " + response);
                return false;
            }
        }
        catch( JsonClientException ex )
        {
            // Response code = 500
            // << {"error":"Failed to close personal account"}
            if ( ex.getResponseCode() == 500 && Utils.maskNull(ex.getData()).contains("close personal") )
            {
                getLogger().error( "accountRemove: error close account? response 500 : " + ex.getData() );
                return false;
            }
            throw ex;
        }
        
        return null;
    }

    @Override
    public Object accountStateModify(AccountOrderEvent accountOrderEvent, ServerContext ctx) throws Exception
    {
        getLogger().info("NetUpOrderManager.accountStateModify->accountModify");
        return accountModify( accountOrderEvent, ctx );
    }

    @Override
    public Object accountOptionsModify(AbstractOrderEvent e, ServerContext ctx) throws Exception
    {
        getLogger().info("NetUpOrderManager.accountOptionsModify->skip");
        return null;
    }

    @Override
    public Object productsModify(ProductOrderEvent productOrderEvent, ServerContext ctx) throws Exception
    {
        getLogger().info("NetUpOrderManager.productsModify");
        
        final TvAccount tvAccount = productOrderEvent.getTvAccount();
        if ( tvAccount == null )
        {
            getLogger().error( "productsModify: productOrderEvent.getTvAccount() = null" );
            return false;
        }

        final String access_card = tvAccount.getDeviceAccountId();
        if (Utils.isBlankString(access_card))
        {
            getLogger().error( "productsModify: access_card for account {id="+tvAccount.getId()+", login=" + tvAccount.getLogin() + "} is empty" );
            return false;
        }
        
        // помимо state проверяем и статус==активе, ибо если неактивный то слать надо пустую подписку (фишка такая, см. камент в accountModify)
        return productsModifySyncFull(productOrderEvent, ctx, access_card, tvAccount.getDeviceState() == TvAccount.STATE_ENABLE && tvAccount.getStatus() == TvAccount.STATUS_ACTIVE );
    }
    
    /**
     * Синхронизация продуктов из orderEvent на access_card Если accountEnabled=false то эта инфа не используется, а
     * только блочится подписка (отправляется пустая).
     */
    private boolean productsModifySyncFull( final AbstractOrderEvent orderEvent, final ServerContext serverContext, final String access_card, final boolean accountEnabled )
        throws Exception
    {
        getLogger().debug( "NetUpOrderManager.productsModifySyncFull access_card=" + access_card + ", accountEnabled=" + accountEnabled );
        
        /*
        по протоколу:
        POST http://mw.iptv/admin/access-media/{access_card}
        {
            "media_contents": [{"media_content_code": 0, "service_type": "tv", "till": 0}],
            "media_groups": [{"media_group_code": 0, "service_type": "tv", "till": 0}]
        }
        */

        JSONObject data = new JSONObject();

        if ( accountEnabled )
        {
            JSONArray media_contents = new JSONArray();
            JSONArray media_groups = new JSONArray();

            // final Set<Long> packagesToAdd = TvDynUtils.getFullSetToEnable( e, o -> Utils.parseLong( o ), serviceMode, true );
            // final Set<Long> packagesToAdd = TvDynUtils.getFullSetToEnable2( e, o -> Utils.parseLong( o ), serviceMode, true );
            // список активных продуктов
            for ( ProductSpec productSpec : orderEvent.getFullProductSpecSetToEnable() )
            {
                getLogger().info( "Product: " + productSpec );
                long code = Utils.parseLong( productSpec.getIdentifier().trim() );
                
                JSONObject spec = new JSONObject();
                spec.put("media_group_code", code);
                spec.put("service_type", "tv");
                // spec.put("till", 0);

                media_groups.put(spec);

                spec = new JSONObject();
                spec.put("media_group_code", code);
                spec.put("service_type", "tvod");
                // spec.put("till", 0);
                
                media_groups.put(spec);
            }

            data.put( "media_contents", media_contents );
            data.put( "media_groups", media_groups );
        }
        
        Object response = jsonClient.invokeAndGetObject( JsonClient.Method.post, requestOptions, conf.providerURL + "/access-media/" + access_card, null, data );
        
        if ( response == null || !(response instanceof Boolean) )
        {            
            getLogger().error("productsModifySyncFull: unknown response: " + response);
            return false;
        }
        
        // >> {"media_groups":[{"service_type":"tv","till":0,"media_group_code":666}],"media_contents":[]}
        // << true
        // >> {"media_groups":[],"media_contents":[]}
        // << true
        
        return ((Boolean)response).booleanValue();
    }
}
